/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   Discrete Output Resource Framework API Header File
::   Copyright   :   (C)2002-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   Dependant upon MPC5xx OS sub-project
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file Resource_DiscreteOut.h
 The BEHAVIOUR_DISCRETE_OUT behaviour provides ON/OFF style control of a module output. The behaviour is
 typically assigned to output resources that are connected to relays and solenoids that have only simple
 control requirements. The output maintains the requested state until requested to change by the
 application. Most output resources support this behaviour. */

#ifndef __RESOURCE_DISCRETEOUT_H
#define __RESOURCE_DISCRETEOUT_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <resource.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/

/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! \brief A discrete output resource's diagnostic information data structure.

     A reference to an instance of this object is supplied when the diagnostic notification pointer described
     by \ref DiscreteOutDiagCbackFuncPtrType is executed. Also see CreateResourceBEHAVIOUR_DISCRETE_OUT() and
     \ref discretediagnostics. */
typedef struct
{
/*! Identifes the resource that generated this diagnostic report instance */
    E_ModuleResource eOwningResource;
/*! The status of the resource identified by S_DiscreteOutDiagInfo::eOwningResource. Often this state
    provides indeterminate information. See \ref discretediagnosticprinciples for further information on this */
    E_ResourceDiagStatus eDiagStatus;
} S_DiscreteOutDiagInfo;

/*! Type describing a read only pointer to a discrete output resource's diagnostic data structure
    \ref S_DiscreteOutDiagInfo */
typedef S_DiscreteOutDiagInfo const* S_DiscreteOutDiagInfoPtr;

/*! \brief Function pointer type that defines the prototype of the diagnostic notification function.

    The framework will call this function periodically, providing the application with diagnostic information
    for the assigned output resource.
    
    The \p in_uAppData is application data that was supplied to the framework via the
    S_DiscreteOutCreateAttributes::uDiagCbackAppData when the callback function was defined. The function is
    defined as part of the CreateResourceBEHAVIOUR_DISCRETE_OUT() call. See \ref howtodiscretediagnostics for
    an example. */
typedef void (*DiscreteOutDiagCbackFuncPtrType)(S_DiscreteOutDiagInfoPtr,NativePtrSizedInt_U in_uAppData);

/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid discrete
           output attributes in the S_DiscreteOutAttributes and S_DiscreteOutCreateAttributes data structures.

Each mask definition applies to a particular member of one of the data structures.
\code
    ...
    S_DiscreteOutAttributes AttribsObj;

    // Going to provide values for the resource condition and report condition so include in the mask
    AttribsObj.uValidAttributesMask = USE_ONESHOT_STATE;
    AttribsObj.eState = RES_ON;
    // The remainder of the data structure can be left undefined since the uValidAttributesMask
    //   indicates that the members are not valid
    ... \endcode */
typedef enum
{
    USE_DISCRETE_DEFAULTS = 0,
    USE_DISCRETE_STATE = 0x01,              /*!< Selects S_DiscreteOutAttributes::eState */
    USE_DISCRETE_CONDITION = 0x02,          /*!< Selects S_DiscreteOutAttributes::eResourceCondition */
    USE_DISCRETE_DIAG_CONDITION = 0x04,     /*!< Selects S_DiscreteOutAttributes::eResourceDiagCondition */
    USE_DISCRETE_DYNAMIC_ON_CREATE = 0x08,  /*!< Selects S_DiscreteOutCreateAttributes::DynamicObj */
    USE_DISCRETE_DIAG_REPORT = 0x10,        /*!< Selects S_DiscreteOutCreateAttributes::pfDiagCback */
    USE_DISCRETE_DRIVER_MODE = 0x20,        /*!< Selects S_DiscreteOutCreateAttributes::eDriverMode */

/* IF THIS TYPE EXCEEDS 0x80 THEN ALTER THE TYPE OF THE uValidAttributesMask MEMBER(S) ACCORDINGLY */

} E_DiscreteOutAttributesMask;

/*! \brief This data structure describes all of the discrete output resource's runtime configuration attributes.

    The attributes are altered through the use of SetResourceAttributesBEHAVIOUR_DISCRETE_OUT(). The data
    structure does not need to be completely filled inorder to be used. The \b uValidAttributesMask is a bit
    field member that is used to identify which attributes are valid. Each attribute is identified with a
    separate bit mask that is logic-ORed into the a mask when an attribute is to be used.
    \code
    ...
    S_DiscreteOutAttributes DiscreteOutObj;

    // Going to affect the eState member and the condition 
    DiscreteOutObj.uValidAttributesMask = USE_DISCRETE_STATE | USE_DISCRETE_CONDITION;  \endcode */    
typedef struct
{
/*! Logic-OR the attributes [\ref USE_DISCRETE_STATE, \ref USE_DISCRETE_CONDITION, \ref USE_DISCRETE_DIAG_CONDITION ]
    that are valid for this this instance of the data structure. */
    uint1 uValidAttributesMask;
/*! The condition of the resource. A disabled resource remains in the safest state, which is OFF, regadless
    of what the other runtime attributes have been set to. Diagnostic reports are also blocked when a resource
    is disabled. Use the \ref USE_DISCRETE_CONDITION mask to enable this attribute.  */
    E_ResourceCond eResourceCondition;
/*! The condition of the resource's diagnostic callback. A disabled callback will not execute, regardless of
    the observed condition.  Disabling the resource will also disable the callback, even if this attribuite is
    set. Use the \ref USE_DISCRETE_DIAG_CONDITION to enable this attribute */
    E_ResourceCond eResourceDiagCondition;
/*! This attribute controls the output state of the resource. Unsigned output resources treat the
    \c RES_ON_FORWARD and \ref RES_ON_REVERSE the same as the ON state \ref RES_ON. Use the
    \ref USE_DISCRETE_STATE to enable this attribute. */
    E_ResourceState eState;
} S_DiscreteOutAttributes;

typedef S_DiscreteOutAttributes const* S_DiscreteOutAttributesPtr;

/*! \brief This data structure describes the creation attributes for a discrete output resource

    Included within the structure is the \b DynamicObj which represents the initial states for the resource's
    dynamic attributes. This data structure is referenced when the resource is created via the
    CreateResourceBEHAVIOUR_DISCRETE_OUT() function and \ref USE_DISCRETE_DYNAMIC_ON_CREATE is included in the
    S_DiscreteOutCreateAttributes::uValidAttributesMask member. */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_DISCRETE_DYNAMIC_ON_CREATE, \ref USE_DISCRETE_DIAG_REPORT,
    and \ref USE_DISCRETE_DRIVER_MODE] that are valid for this this instance of the data structure. */
    uint1 uValidAttributesMask;
/*! The diagnostic notification function that should be used by the framework to notify the application
    of the created resource's diagnostic state. This function will be executed periodically by the framework.
    Select this attribute with the \ref USE_DISCRETE_DIAG_REPORT bit mask. If not selected the default value
    of \c NULL shall be applied. */
    DiscreteOutDiagCbackFuncPtrType pfDiagCback;

/*! Application data that will be supplied as a parameter of the diagnostic notification callback function.
    Its type has been sized to allow a pointer to be stored in this parameter. */
    NativePtrSizedInt_U uDiagCbackAppData;

/*! Initial values of the runtime attributes, which can be later altered through a call to
    SetResourceAttributesBEHAVIOUR_DISCRETE_OUT(). Select with the \ref USE_DISCRETE_DYNAMIC_ON_CREATE bit mask */
    S_DiscreteOutAttributes DynamicObj;

/*! Some drivers have the facility to operate in more than one mode. This attribute allows the mode to be selected.
    Modes include a low side driver and push pull driver. Selected with \ref USE_DISCRETE_DRIVER_MODE, this attribute
    defaults to \ref DRIVER_LOW_SIDE in situations where mixed mode operation is supported or to the appropriate mode
    of the driver hardware when mixed mode operation is not supported. */
    E_DriverMode eDriverMode;
} S_DiscreteOutCreateAttributes;

typedef S_DiscreteOutCreateAttributes const* S_DiscreteOutCreateAttributesPtr;

#pragma pack()
/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/
NativeError_S SetDiscreteOutState(E_ModuleResource, E_ResourceState);

NativeError_S CreateResourceBEHAVIOUR_DISCRETE_OUT(E_ModuleResource, S_DiscreteOutCreateAttributes const*);
NativeError_S DestroyResourceBEHAVIOUR_DISCRETE_OUT(E_ModuleResource eResource);
NativeError_S SetResourceAttributesBEHAVIOUR_DISCRETE_OUT(E_ModuleResource, S_DiscreteOutAttributes const*);

#endif /* __RESOURCE_DISCRETEOUT_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
